/**
*
* \file        tji.c
* \brief       Generic support for Join-Based Tool Interface (TJI)
* \author      Pete McCormick, Adolfo Velasco, Larry Salant
* \date        4/24/2007
* \note        Adapted from Cajita/TPSB project.
*/

///////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "mytji.h"
#include "tji.h"
#include "cresnet_slave.h"
#include "os.h"
#include "memorymanager.h"
#ifdef DEBUG
#include "console.h"
#endif
#include "DMCardJoins.h"
#include "dm_console.h"
#include "Cresnet.h"
#include "dm_string_utils.h"
#include "StreamMgr.h"
#include "dm_errors.h"

///////////////////////////////////////////////////////////////////////////////

const char separator[]={ " ,:\r\n"};

// Pointers to functions for Controller Mode.  They are dynamically linked, if
// device is in controller mode, otherwise, their null
// dynamic linking is done so the controller code is not included for devices that don't need it
PRTJMONITOR_PENDING_DIGITAL_FUNC pfRTJIDigitalPending = 0;
PRTJMONITOR_PENDING_ANALOG_FUNC pfRTJIAnalogPending = 0;
PRTJIMONITOR pfRTJIMonitor = 0;
PTJISET pfTJISetDownStream = 0;
PTJIGET pfTJIGetDownStream = 0;
///////////////////////////////////////////////////////////////////////////////
//
#define MAX_SLOT_CHARS 4
const char HEX_VAL1[23] = {0,1,2,3,4,5,6,7,8,9,0,0,0,0,0,0,0,10,11,12,13,14,15};

#define AsciiToNibble(byt)  (HEX_VAL1[(byt) - '0'])  // it can take lower case hex as well
#define AsciiTohex(str)     ( (AsciiToNibble(*(BYTE *)(str))<<4) | (AsciiToNibble(*(BYTE *)(((BYTE *)(str)) + 1))) )

int TJI_L6_IdentifyFlag(char * pText);
UINT8 getStartEndFlags(int inFlags);

/**
* \author        Aandrew Salmon
* \brief         separate token into steam# and subslot#
* \date          10/10/08
* \param         pText    pointer to string
* \param         pStream  pointer stream result
* \param         pSubslot pointer to subslot result
* \return        Int
* \retval        0 if '.' was found else number of characters for stream data
*
*/
int GetStreamNumSubslotNum( char *pText, UINT8 *pStream, UINT8 *pSubslot  )
{
   char* pText1, *pError;
   UINT32 count =0;
   INT32 length;

   //Verify parameters
   if ( !pText || !pStream || !pSubslot )
       return -1;

   length = strlen(pText);
   pText1 = pText;

   //Bugzilla fix
   //Add support for "DIRECT" command
   if ( !stricmp(pText,"DIRECT") )
   {
       //TJI gives streams in base 1
       *pStream = STREAM_1+1;
       *pSubslot = TOP_LEVEL_SLOT_0;
       return 0;
   }

   while(--length > 0 )
   {
      count++;
      ++pText1;

      // hyphen?
      if (*pText1 == '.')
      {
         // skip the dot, and get the subslot
         ++pText1;

         if(*pText1 == NULL)
            return -1;//no subslot after the "."

         //Get the subslot#
         *pSubslot = strtoul( pText1, &pError, 0 );
         if(*pError)
            return -1;

         strncpy(pText1,pText,count+1);

         //Get the stream#
         *pStream = strtoul(pText1, &pError, 0);
         if(*pError)
            return -1;

         break;
      }
  }

  if( !length )//no separator..hence no subslot
   {
      // find stream number
      *pStream = strtoul(pText, &pError, 0);
      if(*pError)
         return -1;
   }

   if( *pStream < 1 )//must be 1-based
      return -1;

  return length;
}
/**
* \author        Larry Salant
* \brief         Get slot number -
* \param         cmd    - pointer slot number
* \retval        slot #; -1 if invalid
*
*/
INT8 GetSlotNum(char *pSlotNum )
{
  INT8 count;
  char data[MAX_SLOT_CHARS+1]; // allow room for a null
  char* pError;     //Pointer to "strtoul" error

  //Get the slot #.  Currently only support one level (unless we're the controller
  // which is parsed later, if needed)
  // if there are subslots, ignore them for now.
  for (count = 0;  count < MAX_SLOT_CHARS; count++)
  {
    if (*pSlotNum == '.' || *pSlotNum == 0)
      break;
    data[count] = *pSlotNum++;
  }
  data[count] = 0; // make sure its terminated

  count = strtoul(data, &pError, 0);
  if (*pError)
    return -1;

  //Verify the subslot
  if ( !IsValidSubSlotNumber(count) )
    return -1;
  return count;
}
/**
* \author        Adolfo Velasco
* \brief         processes the TJI console get commands
* \date          9/26/06
* \param         cmd    - pointer to command line
* \return        Int
* \retval        number of argumetns
*
*/
int ParseConsoleGetTJI( unsigned char subType, char *RxArgumentsTJI )
{
    unsigned char flags = 0x00;
    unsigned char reqID;
    unsigned char dataCount = 0;
    UINT8 subslot = TOP_LEVEL_SLOT_0;
    char *pText, *pSubSlot=NULL;
    UINT8 *RxBufferTJI;
    char *pTemp;
    unsigned long join;
    char* pError;     //Pointer to "strtoul" error
    INT32 length;

    //Assume no [text] for now...

    //argv contains the TJI delimited text

    //command line should be the TJI delimited text
    // since we have limited RAM, parse input buffer as we go
    // Assume no [text] for now...
    //argv[1]  = <req_id>
    //argv[2]  = <subslot>
    //argv[3]+ = <join_to_get>

    if ( !RxArgumentsTJI || !(*RxArgumentsTJI) || (*RxArgumentsTJI == '?'))                   // print help strilng
    {
#ifdef DEBUG
        DmConsolePrintf("JOINGETxxx {reqid} {stream#.subslot} {join_to_get}\r");
#endif
        return 0;
    }

    // find req ID
    pText = strtok(RxArgumentsTJI,separator);
    if (!pText)
        return -1;
    reqID = strtoul(pText, &pError, 0);
    if (*pError)
        return -1;

    // find subslot number
    pText = strtok(NULL,separator);
    if (!pText)
        return -1;

    //Bugzilla fix
    //Add support for "DIRECT" command
    if ( !stricmp(pText,"DIRECT") )
    {
        subslot = TOP_LEVEL_SLOT_0;
    }
    else
    {
        //Bugzilla (DE1431) fix
        //Get the slot #.  Currently only support one level (unless we're the controller
        // which is parsed later, if needed)
        subslot = GetSlotNum(pText);
        if (!IsValidSubSlotNumber(subslot))
          return -1;

        // save pointer to subslot for controller
        pSubSlot = pText;
    }

    if (!MemMgr || (RxBufferTJI = MemMgr->GetBlock(DM_MAX_USER_CMD_LEN)) == 0)
        return -1;

    //First clear the buffer
    memset( RxBufferTJI, 0, NUM_OF(RxBufferTJI) );

    // get the next token in the command line
    pText = strtok(NULL,separator);
    // for every <join>[-<join>],value pair
    while (pText != NULL)
    {
        // make sure there's room in the buffer
        if (dataCount >= (DM_MAX_USER_CMD_LEN-TJI_PACKET_HEADER_SIZE-1))
        {
            MemMgr->FreeBlock(RxBufferTJI);
            return -1;
        }

        //Get the first join #
        join = strtoul(pText, &pError, 0);
        if (*pError)
            return -1;

        *(UINT16*) &RxBufferTJI[TJI_PACKET_HEADER_SIZE + dataCount] = (UINT16)LE2NATIVE16(join);
        dataCount += 2;

        // get the pointer to the next token
        pTemp = pText;
        pText = strtok(NULL,separator);
        // if this is the last token
        if (!pText)
            // search to end of this token
            length = strlen(pTemp);
        else // search to next token
            length = (UINT8)(pText - pTemp);

        //check if there is a hypen
        while (--length > 0)
        {
            ++pTemp;

            // hyphen?
            if (*pTemp == '-')
            {
                // make sure there's room in the buffer
                if (dataCount >= (DM_MAX_USER_CMD_LEN-TJI_PACKET_HEADER_SIZE-1))
                {
                    MemMgr->FreeBlock(RxBufferTJI);
                    return -1;
                }

                // skip the hyphen, and get end of range
                ++pTemp;
                if (*pTemp == NULL)
                    return -1;

                //Get the next join #
                join = strtoul(pTemp, &pError, 0);
                if (*pError)
                    return -1;

                // set the mod bit to indicate there's a range
                flags |= TJI_FLAG_BIT_MOD;
                // put it in the packet
                *(UINT16*) &RxBufferTJI[TJI_PACKET_HEADER_SIZE + dataCount] = (UINT16)LE2NATIVE16(join);
                dataCount += 2;
                break;
            }
        }
    }

    //Verify parameters
    if (!dataCount)
    {
        MemMgr->FreeBlock(RxBufferTJI);
        return -1;
    }
    // if not a DM Controller or the slot is a local slot
    // otherwise, send the request to the device;  the Controller will print it when it receives
    // the response
    if (!pfTJIGetDownStream ||  (*pfTJIGetDownStream)(subType, pSubSlot, reqID, (char *)&RxBufferTJI[TJI_PACKET_HEADER_SIZE],  dataCount) != 0)
    {
      // not a controller or its a local slot
      //Create the TJI request packet
      RxBufferTJI[0] = CresnetGetDefaultId();     //ID
      RxBufferTJI[1] = TJI_PACKET_HEADER_SIZE - 2 + dataCount;  //Count
      RxBufferTJI[2] = TJI_REQUEST;                             //TJI Request type
      RxBufferTJI[3] = subType;                 //TJI Subtype
      RxBufferTJI[4] = flags;                                       //TJI Subtype
      RxBufferTJI[5] = reqID;                                       //Transaction ID
      //Data has already been populated

      //Send the request
      GetState(GetStreamFromPacket(RxBufferTJI[0]))->ProcessRequestTJI(RxBufferTJI, false, subslot);
    }
    MemMgr->FreeBlock(RxBufferTJI);
    return 0;
}

/**
* \author   Adolfo Velasco, Larry Salant
* \brief    Process the Set Digital command
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinSetDigitalCmd(UINT32 level, char * RxArgumentsTJI)
{
    unsigned long join, value, iMax;
    char* pText, *pTemp, *pSubSlot=NULL;
    char* pError;     //Pointer to "strtoul" error
    unsigned long j;
    UINT8 subslot = TOP_LEVEL_SLOT_0;

    //command line should be the TJI delimited text
    // since we have limited RAM, parse input buffer as we go
    //argv[1]  = [type] (optional)   Assume no [text] for now...
    //argv[1]  = <subslot>
    //argv[2] = <join_to_set>
    //argv[3+] = <digital_value>

    if ( !RxArgumentsTJI || !(*RxArgumentsTJI) || (*RxArgumentsTJI == '?'))                   // print help strilng
    {
#ifdef DEBUG
        DmConsolePrintf("JOINSETDIGITAL [stream#.subslot] join value\r");
#endif
        return 0;
    }

    // find subslot number;  Currently we only support 1 level subslot except if device
    // is a DM Controller and it talks to other devices.
    pText = strtok(RxArgumentsTJI,separator);
    if (!pText)
        return -1;

    //Bugzilla fix
    //Add support for "DIRECT" command
    if ( !stricmp(pText,"DIRECT") )
    {
        subslot = TOP_LEVEL_SLOT_0;
    }
    else
    {
        //Bugzilla (DE1431) fix
        //Get the slot #.  Currently only support one level (unless we're the controller
        // which is parsed later, if needed)
        subslot = GetSlotNum(pText);
         if (!IsValidSubSlotNumber(subslot))
          return -1;

        // save pointer to subslot for controller
        pSubSlot = pText;
   }

    /*
    if ( GetStreamNumSubslotNum( pText, &stream, &subslot ) == -1 )//failed
        return -1;
    */

    // get the next token in the command line
    pText = strtok(NULL,separator);
    // for every <join>[-<join>],value pair
    while (pText != NULL)
    {
        //Get the first join #
        join = strtoul(pText, &pError, 0);
        if (*pError)
            return -1;

        iMax = join;

        // get the pointer to the next token
        pTemp = pText;
        pText = strtok(NULL,separator);
        if (!pText)
            return -1;

        //check if there is a hypen
        while (pTemp++ < pText)
        {
            // hyphen?
            if (*pTemp == '-')
            {
                // skip the hyphen, and get end of range)
                ++pTemp;
                iMax = strtoul(pTemp, &pError, 0);
                if (*pError)
                    return -1;

                //If  bad range, give error
                if ( join > iMax )
                    return -1;
                break;
            }
        }

        //Get the value
        value = strtoul(pText, &pError, 0);
        if (*pError)
            return -1;

        //Check boundaries of value
        if (value > 1)
            return -1;

        //If range, max > join, otherwise list max == join
        for ( j = join; j <= iMax; j++ )
        {
#ifdef DEBUG
            //DmConsolePrintf("JoinSetDigital: slot=%d, join=%d, value=%d\r", slot, j, value);
#endif
            if ( subslot != TOP_LEVEL_SLOT_0 )//if sub-slot
            {
                // if not a DM Controller or the slot is a local slot
                // (otherwise, send it to the device.
                if (!pfTJISetDownStream ||  (*pfTJISetDownStream)(DIGITAL, pSubSlot, j, value) != 0)
                {
                  join = ApplySubSlotMaskToJoinNumber( subslot, j );

                  //Check for an error
                  if ( !IsSubSlottedJoin(join) )
                    continue;
                  // process the join
                  CresnetSlaveProcessInternalDigital(STREAM_1, join, value, 0);
                }
            }
            else
            {
                // process the join
                CresnetSlaveProcessInternalDigital(STREAM_1, j, value, 0);
            }
        }
        // get the next token in the command line
        pText = strtok(NULL,separator);
    }

    return 0;
}

/**
* \author   Adolfo Velasco, Larry Salant
* \brief    Process the Set Analog command
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinSetAnalogCmd(UINT32 level, char * RxArgumentsTJI)
{
    unsigned long join, value, iMax;
    char* pText, *pTemp, *pSubSlot=NULL;
    char* pError;     //Pointer to "strtoul" error
    unsigned long j;
    UINT8 subslot = TOP_LEVEL_SLOT_0;

    //command line should be the TJI delimited text
    // since we have limited RAM, parse input buffer as we go
    //argv[0]  = [type] (optional)   Assume no [text] for now...
    //argv[1]  = <subslot>
    //argv[2] = <join_to_set>
    //argv[3+] = <analog_value>
    if ( !RxArgumentsTJI || !(*RxArgumentsTJI) || (*RxArgumentsTJI == '?'))                   // print help strilng
    {
#ifdef DEBUG
        DmConsolePrintf("JOINSETANALOG [stream#.subslot] join value\r");
#endif
        return 0;
    }

    // find subslot number;  Currently we only support 1 level subslot except if device
    // is a DM Controller and it talks to other devices.
    pText = strtok(RxArgumentsTJI,separator);
    if (!pText)
        return -1;

    //Bugzilla fix
    //Add support for "DIRECT" command
    if ( !stricmp(pText,"DIRECT") )
    {
        subslot = TOP_LEVEL_SLOT_0;
    }
    else
    {
        //Bugzilla (DE1431) fix
        //Get the slot #.  Currently only support one level (unless we're the controller
        // which is parsed later, if needed)
        subslot = GetSlotNum(pText);
         if (!IsValidSubSlotNumber(subslot))
          return -1;

        // save pointer to subslot for controller
        pSubSlot = pText;
    }

    // get the next token in the command line
    pText = strtok(NULL,separator);
    // for every <join>[-<join>],value pair
    while (pText != NULL)
    {
        //Get the first join #
        join = strtoul(pText, &pError, 0);
        if (*pError)
            return -1;
        iMax = join;

        // get the pointer to the next token
        pTemp = pText;
        pText = strtok(NULL,separator);
        if (!pText)
            return -1;

        //check if there is a hypen
        while (pTemp++ < pText)
        {
            // hyphen?
            if (*pTemp == '-')
            {
                // skip the hyphen, and get end of range)
                ++pTemp;
                iMax = strtoul(pTemp, &pError, 0);
                if (*pError)
                    return -1;

                //If  bad range, give error
                if ( join > iMax )
                    return -1;
                break;
            }
        }

        //Get the value
        value = strtoul(pText, &pError, 0);
        if (*pError)
            return -1;

        //Check boundaries of value
        if (value > 65535)
            return -1;

        //If range, max > join, otherwise list max == join
        for ( j = join; j <= iMax; j++ )
        {
#ifdef DEBUG
//      DmConsolePrintf("JoinSetAnalog: slot=%d, join=%d, value=%d\r", slot, j, value);
#endif
            if ( subslot != TOP_LEVEL_SLOT_0 )//if sub-slot
            {
                // if not a DM Controller or the slot is a local slot
                // (otherwise, send it to the device.
                if (!pfTJISetDownStream ||  (*pfTJISetDownStream)(NEW_ANALOG, pSubSlot, j, value) != 0)
                {
                  // process it locally
                  join = ApplySubSlotMaskToJoinNumber( subslot, j );

                  //Check for an error
                  if ( !IsSubSlottedJoin(join) )
                    continue;

                  // process the join
                  CresnetSlaveProcessInternalAnalog(STREAM_1, join, value, 0);
                }
            }
            else
            {
                // process the join
                CresnetSlaveProcessInternalAnalog(STREAM_1, j, value, 0);
            }
        }
        // get the next token in the command line
        pText = strtok(NULL,separator);
    }

    return 0;
}

/**
* \author         Adolfo Velasco
*
* \brief          Parses string and converts all escapes sequence text into proper ASCII characters
*                 and stores the new string
* \detail
* \date           9/26/06
*
*  \param         string    - string to parse
*                 newstring - new string w/translated escape sequences
*                 size      - size of new string
*
* \warning        Use return value for length, do not use strlen on returned string.
*
*  \return        int
*
*  \retval        length of new string (This is required because
*  				  the translated string may contain unescaped
*  				  null characters.)
*
*/
int TranslateEscapes(char* string, char* newstring, int size)
{
  //int errorcode = 0;
  int i, j, length; //j;
  unsigned char temp;
  INT8* tptr;

#ifdef DEBUG
  //DmConsolePrintf("TranslateEscapes: string=%s, size=%d", string, size);
#endif

  //Empty the new string
  //newstring.Empty();
  memset(newstring, 0, size);

  length = strlen(string);

  for(i = 0, j = 0; (i < length) && (j < size); i++)
  {
    temp = string[i];

    // check to see if this character starts an escape...
    if(string[i] == '\\')
    {
      // verify that the '\' is not at the end of the string... if it is, error
      if( !(i+2 > length))
      {
        switch (string[i+1])
        {
        case 0x22:
          newstring[j++] = 0x22;    //Double quote
          i++;
          break;

        case 'a':
          newstring[j++] = 0x07;    //Audible bell
          i++;
          break;

        case 'b':
          newstring[j++] = 0x08;    //Backspace
          i++;
          break;

        case 'f':
          newstring[j++] = 0x0C;    //Form feed
          i++;
          break;

        case 'n':
          newstring[j++] = 0x0D;    //Carriage return
          //newstring[j++] = 0x0A;  //Line feed - DVPHD does not recognize this character
          i++;
          break;

        case 'r':
          newstring[j++] = 0x0D;    //Carriage return
          i++;
          break;

        case 't':
          newstring[j++] = 0x09;    //Horizontal tab
          i++;
          break;;

        case 'v':
          newstring[j++] = 0x0B;    //Vertical tab
          i++;
          break;

        case '\\':
          newstring[j++] = 0x5C;    //Backslash
          i++;
          break;

          //Hexadecimal character
        case 'X':
        case 'x':
          // Can we grab enough characters before processing the number?
          if ((i+4) > length)
          {
            newstring[j++] = temp;
          }
          else
          {
            //  next should be 7f: \x01\x02
            tptr = (INT8*)&string[i+2];
            newstring[j++] = AsciiTohex(tptr);

             i+=3; // make sure to start at next character...
          }
          break;

        default:
          newstring[j++] = temp;
          break;
        }
      }
      else
      {
        // error, a \ mark was found but at the end of the string.
		// bug fix - a \ should not be part of the string.
        newstring[j++] = temp;
      }
    }
    else
    {
      // not part of an escape, add it to the string...
      newstring[j++] = temp;
    }
  }
  // return length of new string
  return j;
}
/**
* \author   Adolfo Velasco, Larry Salant
* \brief    Process the Set Serial command
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinSetSerialCmd(UINT32 level, char * RxArgumentsTJI)
{
    unsigned long join;
    INT8 subslot = TOP_LEVEL_SLOT_0;
    char* pParse;
    char* pText, *pSubSlot=NULL;      //Pointer to string from command line
    char* pError;     //Pointer to "strtoul" error
    char stText[DM_MAX_USER_CMD_LEN]; //Buffer to hold input string
    UINT32 lStringOffset = 0;
    int lSerialJoinLen;
    int lSerialJoinFlags = 0;
    UINT8 flags;

    //command line should be the TJI delimited text
    // since we have limited RAM, parse input buffer as we go
    //argv[0]  = [type] (optional)   Assume no [text] for now...
    //argv[1]  = [flags]    (Beacuse flags are delimited by space and there could be up to two, the indexes from here on could add +1)
    //argv[2]  = <subslot>
    //argv[3] = <join_to_set>
    //argv[4+] = <analog_value>

    if ( !RxArgumentsTJI || !(*RxArgumentsTJI) || (*RxArgumentsTJI == '?'))                   // print help strilng
    {
#ifdef DEBUG
        DmConsolePrintf("JOINSETSERIAL [stream#.subslot#] join string\r");
#endif
        return 0;
    }

    //Extract Flags if any
    pText = strtok(RxArgumentsTJI,separator);
    if (!pText)
        return -1;

    if(pText[0] == '!')
    {
       lSerialJoinFlags |= TJI_L6_IdentifyFlag(pText);
       //Update the string offset
       lStringOffset += strlen(pText) + 1;

      //There could be a second Flag in the same command, just before the Slot
      pText = strtok(NULL,separator);
      if (!pText)
        return -1;

      if(pText[0] == '!')
      {
       lSerialJoinFlags |= TJI_L6_IdentifyFlag(pText);
       //Update the string offset
       lStringOffset += strlen(pText) + 1;

       pText = strtok(NULL,separator);
       if (!pText)
        return -1;
      }
    }

    //Bugzilla fix
    //Add support for "DIRECT" command
    if ( !stricmp(pText,"DIRECT") )
    {
        subslot = TOP_LEVEL_SLOT_0;
    }
    else
    {
        //Bugzilla (DE1431) fix
        //Get the slot #.  Currently only support one level (unless we're the controller
        // which is parsed later, if needed)
        subslot = GetSlotNum(pText);
         if (!IsValidSubSlotNumber(subslot))
            return -1;

        pSubSlot = pText;
    }

    //Update the string offset
    lStringOffset += strlen(pText) + 1;

    // get the next token in the command line
    pText = strtok(NULL,separator);
    if (!pText)       //Check for invalid character
        return -1;

    //Get the join #
    join = strtoul(pText, &pError, 0);
    if (*pError)
        return -1;

    //Update the string offset
    lStringOffset += strlen(pText) + 1;

    //<string>
    pParse = &RxArgumentsTJI[lStringOffset];
    memset( &stText[0], 0, NUM_OF(stText) );  //Clear the string
    pText = &stText[0];                       //Start of text buffer

    //Check for literals
    if ( *pParse == '"' )
    {
        //Make sure more characters exist, otherwise this is a clear command
        if ( strlen(pParse) > 1 )
        {
            pParse++;
            _strncpy( pText, pParse, NUM_OF(stText) );
        }
		lSerialJoinLen = strlen(pText);
    }
    else //Process the escapes
    {
        lSerialJoinLen = TranslateEscapes( pParse, pText, NUM_OF(stText) );
    }
    //String text has been processed and ready to be sent
#ifdef DEBUG
    //DmConsolePrintf("JoinSetSerialCmd: slot=%d, join=%d, value=%s", slot, join, pText);
#endif
    if ( subslot != TOP_LEVEL_SLOT_0 )//if sub-slot
    {
        // if not a DM Controller or the slot is a local slot
        // (otherwise, send it to the device.
        if (!pfTJISetDownStream ||  (*pfTJISetDownStream)(EXTENDED_SERIAL, pSubSlot, join, (UINT32)pText) != 0)
        {
            join = ApplySubSlotMaskToJoinNumber( subslot, join );

            //Check for an error
            if ( !IsSubSlottedJoin(join) )
                return -1;
        }
        else
          return 0;
    }


    flags = getStartEndFlags(lSerialJoinFlags);

    //Apply the new serial  (for now, use slot as stream
    CresnetSlaveProcessInternalSerial(STREAM_1, join, pText, lSerialJoinLen, 0 ,flags);
    return 0;
}

/**
* \author   Luis Rivera
* \brief    Will return the TJI Level 6 flag's bit position
*           value, will return 0 if not identified.
* \date
* \param    level - command level
* \param
* \return   int
* \retval
*/
int TJI_L6_IdentifyFlag(char * pText)
{
    if(!pText)
      return 0;

    if(!stricmp(pText,"!LAST"))
    {
      return FIRST_FLAG_ACTIVE;
    }

    if(!stricmp(pText,"!FIRST"))
    {
      return LAST_FLAG_ACTIVE;
    }

    return 0;
}


/**
* \author   Luis Rivera
* \brief    Translates the TJI L6 Multipart Serial Join Flags
*           into the dm_cresnet_slave.h Start/End flags
* \date
* \param    level - command level
* \param
* \return   int
* \retval
*/
UINT8 getStartEndFlags(int inFlags)
{
  if(inFlags == 0)
  {
      return START_AND_END;
  }
  if(inFlags == LAST_FLAG_ACTIVE)
  {
      return APPEND_AND_END;
  }
  else if(inFlags == FIRST_FLAG_ACTIVE)
  {
      return START_NOT_END;
  }
  else if(inFlags == (FIRST_FLAG_ACTIVE + LAST_FLAG_ACTIVE))
  {
      return APPEND_NOT_END;
  }

  return 0;
}



/**
* \author   Andrew Salmon
* \brief    Get dynamic memory and consolidate tji string up to 80 bytes and send
* \date     8/5/11
* \param    pBegin - source data
* \param    count - number of data bytes in response packet w/ <id> stripped
* \param    rtji - is rtji
* \return   int
* \retval   0 = okay
*/
void SendResponseTJI( UINT8 *pBegin, UINT8 count , BOOL rtji)
{
    char *rtjiFormatted, *rtjiFormattedData;
    UINT32  k, dataFormatted;

    //Calculate how the data will be broken down into packets
    //restrict to medium buffer size MEDIUM_BLOCK_LEN = 80
    // + 5 for 2 header (0xFA, 0xC1) bytes + 3 footer (0xFB\r'\0') bytes
    rtjiFormatted = (char *)GetCresnetBuffer( min( MEDIUM_BLOCK_LEN, ( 2* count) + 5 ) );

     if(rtjiFormatted)
     {
          // lock the console until tji complete
          if(DmConsoleTjiPrintLock())
          {
              if( rtji )//source =RTJI
                 //RTJI spec according to Bill Levine
                 sprintf(&rtjiFormatted[0], "%c%c", 0xFA,0xC1) ;
              else
                 //TJI spec v2 - Fixed header
                 sprintf(&rtjiFormatted[0], "%c%c", 0xFA,0xC0) ;

              rtjiFormattedData = &rtjiFormatted[2];
              dataFormatted = 2; // 2 header bytes

              //Response packet w/ <id> stripped
              for ( k = 1; k < count; k++ )
              {
                  sprintf(rtjiFormattedData, "%02x", *(pBegin+k)) ;

                  dataFormatted += 2;

                  rtjiFormattedData += 2;
                  //TJI_PACKET_MAX_TOTAL_SIZE

                  // -2 so sprintf do not write '\0' outside of the block
                  if (((dataFormatted >= (MEDIUM_BLOCK_LEN-2) )) && ((dataFormatted % (MEDIUM_BLOCK_LEN-2) ) == 0 ) ||
                      ((MEDIUM_BLOCK_LEN > TJI_PACKET_MAX_TOTAL_SIZE ) && (dataFormatted >= (TJI_PACKET_MAX_TOTAL_SIZE-2)) && (( dataFormatted % (TJI_PACKET_MAX_TOTAL_SIZE-2) ) == 0 )))
                  {
                      //send paket and move pointer to begining of block
                      DmConsolePrintf( "%s", &rtjiFormatted[0] );
                      rtjiFormattedData = &rtjiFormatted[0];
                  }

              }

              //Finish the packet
              sprintf(rtjiFormattedData, "%c\r", 0xFB) ;

              //send paket and start at begining
              DmConsolePrintf( "%s", &rtjiFormatted[0] );

              // unlock for next task
              DmConsoleTjiPrintUnlock();
          }

          ReleaseCresnetBuffer((UINT8*)rtjiFormatted);
     }
}
/**
* \author      Adolfo Velasco, Larry Salant
* \brief       Creates the TJI response packet and sends out Cresnet
* \date        9/18/06
* \param       subType      - TJI type
* \param       bFlag        -
* \param       transID      - Transaction ID
* \param       pData        - Pointer to data array
* \param       pJoinHeader  - Pointer to header - if not NULL, header has 2 bytes with the join number
* \param       dataCount    - # of bytes in data array including header, if any
* \param       bMore        - Flag to tell if MORE bit should be set on all packets
* \param       bCresnet     - 0 if console, otherwise where request came from Cresent or CIP
* \param       bStream      - stream that the join belongs to
*  \return     None
*  \retval     None
*
*/
void ProcessResponseTJI( UINT8 subType, UINT8 bFlag, UINT8 transID, UINT8* pData, UINT8* pJoinHeader,
                         UINT32 dataCount, BOOL bMore, UINT8 bCresnet, UINT8 bStream, BOOL rtji/* default to false*/)
{
    UINT32 i, j;
    UINT8* pPacket, *pBegin;
    UINT32 lPacketSize;
    UINT32 lTotalPackets = 0;

    //Calculate how the data will be broken down into packets
    lPacketSize = min( TJI_PACKET_DATA_MAX_SIZE, dataCount );        //Data is limited by buffer size

    //get a pointer to the scratchpad buffer
    pPacket = GetCresnetBuffer( lPacketSize + TJI_PACKET_HEADER_SIZE );
    pBegin = pPacket;

    if( pPacket )
    {
        *pPacket++ = TJI_RESP_DEST_ID;
        *pPacket++ = 0;                     //Leave the packet length blank for now
        *pPacket++ = TJI_RESPONSE;
        *pPacket++ = subType;
        *pPacket++ = bFlag;
        *pPacket++ = transID;

        //Function allows (pData == NULL) so make sure lengths are zero if that's the case
        if ( !pData )
        {
            dataCount = 0;
            lPacketSize = 0;
        }

        //Check if the join header exists since it will need to be transmitted every packet
        if (pJoinHeader != NULL)
        {
            if (!dataCount)
            {
                //Null serials have no data but join # must be returned
                lTotalPackets = 1;
            }
            else
            {
                //Calculate how many packets will be sent
                lTotalPackets = dataCount / lPacketSize;

                //Check for a remaining packet
                if (dataCount % lPacketSize)
                {
                    lTotalPackets++;
                }
            }

            //Each packet requires a 2 byte join header so add it
            dataCount += (lTotalPackets * 2);

            //Recalculate the packet size
            lPacketSize = min( TJI_PACKET_DATA_MAX_SIZE, dataCount );
        }

        //Set the pointer to the data section
        pPacket = pBegin + TJI_PACKET_HEADER_SIZE;
        j = 0;

        //Continue sending packets until all the data has been sent
        i = 0;
        while (i < dataCount)
        {
            //To eliminate copying large serials, the header, if present contains the join number
            if (pJoinHeader != NULL)
            {
                // put the join number into the buffer
                *pPacket++ = pJoinHeader[0];
                *pPacket++ = pJoinHeader[1];
                j = 2;
            }

            //Bugzilla fix
            //Function now allows (pData == NULL) so only parse data if needed
            if ( pData )
            {
                //Populate the data section until full
                while ((j < lPacketSize) && ( (i+j) < dataCount ))
                {
                    *pPacket++ = *pData++;
                    j++;
                }
            }

            //Recalculate the packet length
            pBegin[TJI_HEADER_LENGTH-1] = (j + TJI_PACKET_HEADER_SIZE - 2);

            //Set the MORE bit in the flag byte if necessary
            if ( bMore || (i+j) < dataCount )
                *(pBegin+TJI_HEADER_FLAG) |= TJI_FLAG_BIT_MORE;
            else
                *(pBegin+TJI_HEADER_FLAG) &= ~(TJI_FLAG_BIT_MORE);    //Last packet!

            //Send out the TJI packet
            if ( bCresnet )
            {
                UINT16 timeout = CRESNET_QUEUE_TIMEOUT_10MS;

                // always wait for room in the queue (bug fix for tji response getting truncated)
                while (IsCrestnetSlaveQueueFull(bStream, bCresnet) && --timeout )
                    HwDelayMsec(10);

                // send packet to queue
                if( timeout > 0 )
                    CresnetSlaveEnqueueOutgoingPacket(bStream, pBegin, TJI_PACKET_HEADER_SIZE + j, bCresnet);
                else
                    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_DROP_TJI_SENDING, DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
            }
            else //Output to console
            {
            	 SendResponseTJI( pBegin, (j + TJI_PACKET_HEADER_SIZE) , rtji);
            }

            //Reset the pointer to the data section
            pPacket = pBegin + TJI_PACKET_HEADER_SIZE;
            i += lPacketSize;
            j = 0;
        }

        if( !bCresnet && rtji )
           DmConsoleShowPrompt();

        // free up temporary buffer
        ReleaseCresnetBuffer(pBegin);
    }
    else
    {
       if( !bCresnet && rtji )
       		DmConsoleShowPrompt();
    }



}

/**
* \author   Larry Salant
* \brief    Process the Get Analog In Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetInAnalogCmd(UINT32 level, char * RxArgumentsTJI)
{
  return ParseConsoleGetTJI(TJI_SUBTYPE_ANALOG_INPUT, RxArgumentsTJI);
}

/**
* \author   Larry Salant
* \brief    Process the Get Digital In Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetInDigitalCmd(UINT32 level, char * RxArgumentsTJI)
{
  return ParseConsoleGetTJI(TJI_SUBTYPE_DIGITAL_INPUT, RxArgumentsTJI);
}

/**
* \author   Larry Salant
* \brief    Process the Get Serial In Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetInSerialCmd(UINT32 level, char * RxArgumentsTJI)
{
  //int arguments = ParseConsoleCommandTJI(cmd);
  return ParseConsoleGetTJI(TJI_SUBTYPE_SERIAL_INPUT, RxArgumentsTJI);
}

/**
* \author   Larry Salant
* \brief    Process the Get Analog Output Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetOutAnalogCmd(UINT32 level, char * RxArgumentsTJI)
{
  //int arguments = ParseConsoleCommandTJI(cmd);
  return ParseConsoleGetTJI(TJI_SUBTYPE_ANALOG_OUTPUT, RxArgumentsTJI);
}

/**
* \author   Larry Salant
* \brief    Process the Get Digital Output Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetOutDigitalCmd(UINT32 level, char * RxArgumentsTJI)
{
  //int arguments = ParseConsoleCommandTJI(cmd);
  return ParseConsoleGetTJI(TJI_SUBTYPE_DIGITAL_OUTPUT, RxArgumentsTJI);
}

/**
* \author   Larry Salant
* \brief    Process the Get Serial Output Join
* \date     5/9/08
* \param    level - command level
* \param    RxArgumentsTJI - pointer to command string
* \return   int
* \retval   0 = okay
*/
INT32 JoinGetOutSerialCmd(UINT32 level, char * RxArgumentsTJI)
{
  //int arguments = ParseConsoleCommandTJI(cmd);
  return ParseConsoleGetTJI(TJI_SUBTYPE_SERIAL_OUTPUT, RxArgumentsTJI);
}



